<?php

/*
 * @license Commercial
 * @author info@ocdemo.eu
 * 
 * All code within this file is copyright OC Mega Extensions.
 * You may not copy or reuse code within this file without written permission.
*/

class MRewardPointsHelper {
	
	/**
	 * @var object
	 */
	private $_ctrl;
	
	/**
	 * @param object $ctrl
	 * @return \self
	 */
	public static function make( & $ctrl ) {
		return new self( $ctrl );
	}
	
	/**
	 * @param object $ctrl
	 */
	public function __construct( & $ctrl ) {
		$this->_ctrl = & $ctrl;
	}
	
	/**
	 * @param string $reason_code
	 * @return int
	 */
	public function getTotalCustomerRewardsByReasonCode( $reason_code, $customer_id ) {
		return $this->_ctrl->db->query("SELECT COUNT(*) AS `total` FROM `" . DB_PREFIX . "customer_reward` WHERE `customer_id` = '" . (int)$customer_id . "' AND `reason_code` = '" . $this->_ctrl->db->escape( $reason_code ) . "'")->row['total'];
	}
	
	/**
	 * @param int $order_id
	 * @return int
	 */
	public function getTotalCustomerRewardsByOrderId($order_id) {
		return $this->_ctrl->db->query("SELECT COUNT(*) AS `total` FROM `" . DB_PREFIX . "customer_reward` WHERE `order_id` = '" . (int)$order_id . "'")->row['total'];
	}
	
	public function getTotalCustomerCompleteOrders( $customer_id ) {
		/* @var $config array */
		if( null == ( $config = $this->_ctrl->config->get( 'mreward_points_settings' ) ) || empty( $config['order_status_add'] ) ) {
			return 0;
		}
		
		return $this->_ctrl->db->query("SELECT COUNT(*) as `total` FROM `" . DB_PREFIX . "order` WHERE `customer_id` = '" . (int)$customer_id . "' AND `order_status_id` IN (" . implode(',', $config['order_status_add']) . ")")->row['total'];
	}
	
	// @since OC 2.0.2.0 ///////////////////////////////////////////////////////
	
	/**
	 * @since OC 2.0.2.0
	 * @param int $customer_id
	 * @return int
	 */
	public function getRewardPoints($customer_id) {
		return $this->_ctrl->db->query("SELECT SUM(`points`) AS `total` FROM `" . DB_PREFIX . "customer_reward` WHERE `customer_id` = '" . (int)$customer_id . "'")->row['total'];
	}
	
	/**
	 * @param int $order_status_id
	 * @return array
	 */
	public function getOrderStatus($order_status_id) {
		return $this->_ctrl->db->query("SELECT * FROM `" . DB_PREFIX . "order_status` WHERE `order_status_id` = '" . (int)$order_status_id . "' AND `language_id` = '" . (int)$this->_ctrl->config->get('config_language_id') . "'")->row;
	}
	
	// end @since OC 2.0.2.0 ///////////////////////////////////////////////////
	
	public function autoGeneratePoints( $product_id ) {
		/* @var $settings array */
		$settings = $this->_ctrl->config->get('mreward_points_settings');
		
		if( ! empty( $settings['auto_generate_points'] ) ) {
			$this->generatePoints(array(
				'product_id' => $product_id,
			), $settings['auto_generate_points']);
		}
	}
	
	public function generatePoints( array $filter = array(), $auto_generate = false ) {
		/* @var $config array */
		$config = $this->_ctrl->config->get('mreward_points_generator');
		
		/* @var $file_data string */
		$file_data = DIR_CACHE . 'mreward_points.tmp';
		
		/* @var $steps array */
		$steps = array( 'products', 'rewards', 'discounts', 'specials' );
		
		/* @var $data array */
		$data = ! $auto_generate && file_exists( $file_data ) ? json_decode( file_get_contents( $file_data ), true ) : array(
			'steps' => $steps,
			'page' => 1,
			'limit' => 100,
		);
		
		/* @var $step string */
		$step = array_shift( $data['steps'] );
		
		/* @var $conditions array */
		$conditions = array(
			"`p`.`skip_rewards_generator` != '1'"
		);
		
		if( isset( $filter['product_id'] ) ) {
			$conditions[] = '`p`.`product_id` = ' . (int) $filter['product_id'];
		}

		// PRODUCTS
		if( $auto_generate || $step == 'products' ) {
			/* @var $copy_conditions array */
			$copy_conditions = $conditions;
			
			if( $auto_generate && $auto_generate == 'if_empty' ) {
				$copy_conditions[] = '`points` = 0';
			}
			
			$this->_ctrl->db->query("
				UPDATE
					`" . DB_PREFIX . "product` AS `p`
				SET
					`points` = " . ( $config['points'] <= 0 ? '0' : "FLOOR( `price` * 100 / " . $config['points'] . " )" ) . "
				WHERE
					" . implode( ' AND ', $copy_conditions )
			);
			
			$data['total'] = $data['limit'];
		}
		
		// REWARDS
		if( $auto_generate || $step == 'rewards' ) {		
			if( ! $auto_generate && ! isset( $data['total'] ) ) {
				$data['total'] = $this->_ctrl->db->query("SELECT COUNT(*) AS `total` FROM `" . DB_PREFIX . "product` AS `p` WHERE " . implode( ' AND ', $conditions ) )->row['total'];
			}
		
			$query = $this->_ctrl->db->query("
				SELECT
					`p`.`product_id`,
					`p`.`price`
				FROM
					`" . DB_PREFIX . "product` AS `p`
				WHERE
					" . implode( ' AND ', $conditions ) . "
				LIMIT
					" . ( ( $data['page'] - 1 ) * $data['limit'] ) . ", " . $data['limit']
			);
			
			/* @var $product_ids array */
			$product_ids = array();
			
			foreach( $query->rows as $row ) {
				$product_ids[] = $row['product_id'];
			}
			
			/* @var $rewards array */
			$rewards = array();
			
			if( $product_ids ) {
				/* @var $row array */
				foreach( $this->_ctrl->db->query("SELECT * FROM `" . DB_PREFIX . "product_reward` AS `pr` WHERE `product_id` IN(" . implode( ',', $product_ids ) . ")")->rows as $row ) {
					$rewards[$row['product_id'].'_'.$row['customer_group_id']] = $row['points'];
				}
			}

			/* @var $row array */
			foreach( $query->rows as $row ) {
				/* @var $customer_group_id int */
				/* @var $value float */
				foreach( $config['groups'] as $customer_group_id => $value ) {
					if( $auto_generate && $auto_generate == 'if_empty' && ! empty( $rewards[$row['product_id'].'_'.$customer_group_id] ) ) {
						continue;
					}
					
					/* @var $points float */
					$points	= $config['amount'] > 0 ? $row['price'] / $config['amount'] : 0;
					$points	= floor( $points * $value );
					
					if( $points ) {
						if( isset( $rewards[$row['product_id'].'_'.$customer_group_id] ) ) {
							$this->_ctrl->db->query("
								UPDATE
									`" . DB_PREFIX . "product_reward`
								SET
									`points` = '" . $points . "'
								WHERE
									`product_id` = '" . $row['product_id'] . "' AND `customer_group_id` = '" . $customer_group_id . "'
							");
						} else {
							$this->_ctrl->db->query("
								INSERT INTO
									`" . DB_PREFIX . "product_reward`
								SET
									`product_id` = '" . $row['product_id'] . "',
									`customer_group_id` = '" . $customer_group_id . "',
									`points` = '" . $points . "'
							");
						}
					} else {
						$this->_ctrl->db->query("
							DELETE FROM
								`" . DB_PREFIX . "product_reward`
							WHERE
								`product_id` = '" . $row['product_id'] . "' AND `customer_group_id` = '" . $customer_group_id . "'
						");
					}
				}
			}
		}
			
		// DISCOUNTS
		if( $auto_generate || $step == 'discounts' ) {
			/* @var $copy_conditions */
			$copy_conditions = $conditions;
			
			if( $auto_generate && $auto_generate == 'if_empty' ) {
				$copy_conditions[] = '`pd`.`reward_points` = 0';
			}
			
			/* @var $sql string */
			$sql = "
				SELECT
					{columns}
				FROM
					`" . DB_PREFIX . "product_discount` `pd`
				LEFT JOIN
					`" . DB_PREFIX . "product` AS `p`
				ON
					`pd`.`product_id` = `p`.`product_id`
				WHERE
					" . implode( ' AND ', $copy_conditions );
			
			if( ! $auto_generate && ! isset( $data['total'] ) ) {
				$data['total'] = $this->_ctrl->db->query(str_replace(array(
					'{columns}',
				), array(
					'COUNT(DISTINCT `p`.`product_id`) AS `total`',
				), $sql))->row['total'];
			}
			
			$query = $this->_ctrl->db->query(str_replace(array(
				'{columns}',
			), array(
				'`pd`.`product_discount_id`,
				 `pd`.`price`,
				 `pd`.`customer_group_id`'
			), $sql) . ( $auto_generate ? '' : "
				LIMIT
					" . ( ( $data['page'] - 1 ) * $data['limit'] ) . ", " . $data['limit']
			));
			
			/* @var $row array */
			foreach( $query->rows as $row ) {				
				/* @var $value float */
				$value = isset( $config['groups'][$row['customer_group_id']] ) ? $config['groups'][$row['customer_group_id']] : 0;
				
				/* @var $points float */
				$points	= $config['amount'] > 0 ? $row['price'] / $config['amount'] : 0;
				$points	= floor( $points * $value );
				
				$this->_ctrl->db->query("
					UPDATE
						`" . DB_PREFIX . "product_discount`
					SET
						`reward_points` = '" . $points . "'
					WHERE
						`product_discount_id` = '" . $row['product_discount_id'] . "'
				");
			}
		}
			
		// SPECIALS
		if( $auto_generate || $step == 'specials' ) {
			/* @var $copy_conditions */
			$copy_conditions = $conditions;
			
			if( $auto_generate && $auto_generate == 'if_empty' ) {
				$copy_conditions[] = '`ps`.`reward_points` = 0';
			}
			
			/* @var $sql string */
			$sql = "
				SELECT
					{columns}
				FROM
					`" . DB_PREFIX . "product_special` AS `ps`
				LEFT JOIN
					`" . DB_PREFIX . "product` AS `p`
				ON
					`ps`.`product_id` = `p`.`product_id`
				WHERE
					" . implode( ' AND ', $copy_conditions );
			
			if( ! $auto_generate && ! isset( $data['total'] ) ) {
				$data['total'] = $this->_ctrl->db->query(str_replace(array(
					'{columns}',
				), array(
					'COUNT(DISTINCT `p`.`product_id`) AS `total`',
				), $sql))->row['total'];
			}
			
			$query = $this->_ctrl->db->query(str_replace(array(
				'{columns}',
			), array(
				'`ps`.`product_special_id`,
				`ps`.`price`,
				`ps`.`customer_group_id`'
			), $sql) . (  $auto_generate ? '' : "
				LIMIT
					" . ( ( $data['page'] - 1 ) * $data['limit'] ) . ", " . $data['limit']
			));

			/* @var $row array */
			foreach( $query->rows as $row ) {				
				/* @var $value float */
				$value = isset( $config['groups'][$row['customer_group_id']] ) ? $config['groups'][$row['customer_group_id']] : 0;

				$points	= $config['amount'] > 0 ? $row['price'] / $config['amount'] : 0;
				$points	= floor( $points * $value );
				
				$this->_ctrl->db->query("
					UPDATE
						`" . DB_PREFIX . "product_special`
					SET
						`reward_points` = '" . $points . "'
					WHERE
						`product_special_id` = '" . $row['product_special_id'] . "'
				");
			}
		}
		
		if( $auto_generate ) {
			return;
		}
		
		$data['pages'] = ceil( $data['total'] / $data['limit'] );
		
		if( $data['page'] < $data['pages'] ) {
			$data['page']++;
			array_unshift( $data['steps'], $step );
		} else {
			$data['page'] = 1;
			unset( $data['total'] );
			unset( $data['pages'] );
		}
			
		if( $data['steps'] ) {
			file_put_contents( $file_data, json_encode( $data ) );
		} else if( file_exists( $file_data ) ) {
			unlink( $file_data );
		}
		
		return $data['steps'] ? array(
			'step' => array_search( $step, $steps ) + 1,
			'steps' => count( $steps ),
			'progress' => isset( $data['pages'] ) ? round( ( $data['page'] - 1 ) / $data['pages'] * 100, 2 ) : 100,
		) : array(
			'step' => count( $steps ),
			'steps' => count( $steps ),
			'progress' => 100,
		);
	}

	/**
	 * @param int $order_id
	 * @return int
	 */
	public function getTotalRewardsFromOrderProductsByOrderId($order_id) {
		/* @var $reward int */
		$reward = 0;

		/* @var $order_product_query */
		$order_product_query = $this->_ctrl->db->query("SELECT * FROM `" . DB_PREFIX . "order_product` WHERE `order_id` = '" . (int)$order_id . "'");

		/* @var $product array */
		foreach( $order_product_query->rows as $product ) {
			$reward += $product['reward'];
		}

		return $reward;
	}

	public function addReward($customer_id, $description = '', $points = '', $order_id = 0) {
		$this->_ctrl->db->query("INSERT INTO " . DB_PREFIX . "customer_reward SET customer_id = '" . (int)$customer_id . "', order_id = '" . (int)$order_id . "', points = '" . (int)$points . "', description = '" . $this->_ctrl->db->escape($description) . "', date_added = NOW()");

		$mrp_settings	= $this->_ctrl->config->get( 'mreward_points_settings' );

		if( ! empty( $mrp_settings['email_notification_status'] ) && ! empty( $points ) ){			
			$order_info		= $this->_ctrl->model_checkout_order->getOrder($order_id);
			$lang			= (int)$this->_ctrl->config->get('config_language_id');
			$order_lang		= isset($order_info['language_id']) ? $order_info['language_id'] : $lang;

			$subject = $mrp_settings['notify_content'][$order_lang]['notify_title'];

			$find = array(
				'{order_id}',
				'{reward_points}',
				'{reward_points_total}',
				'{first_name}',
				'{last_name}'
			);

			$replaceTo = array(
				'order_id'				=> $order_id,
				'reward_points'			=> $points,
				'reward_points_total'	=> $this->getRewardPoints($customer_id),
				'first_name'			=> $order_info['firstname'],
				'last_name'				=> $order_info['lastname']
			);

			$message = $mrp_settings['notify_content'][$order_lang]['description'];
			$message = str_replace(array("\r\n", "\r", "\n"), '<br />', preg_replace(array("/\s\s+/", "/\r\r+/", "/\n\n+/"), '<br />', trim(str_replace($find, $replaceTo, $message))));

			$store_name = isset($order_info['store_name']) ? $order_info['store_name'] : $this->_ctrl->config->get('config_name');
		
			if( version_compare( VERSION, '3', '>=' ) ) {
				$mail = new Mail($this->_ctrl->config->get('config_mail_engine'));
				$mail->parameter = $this->_ctrl->config->get('config_mail_parameter');
				$mail->smtp_hostname = $this->_ctrl->config->get('config_mail_smtp_hostname');
				$mail->smtp_username = $this->_ctrl->config->get('config_mail_smtp_username');
				$mail->smtp_password = html_entity_decode($this->_ctrl->config->get('config_mail_smtp_password'), ENT_QUOTES, 'UTF-8');
				$mail->smtp_port = $this->_ctrl->config->get('config_mail_smtp_port');
				$mail->smtp_timeout = $this->_ctrl->config->get('config_mail_smtp_timeout');
			} else if( null != ( $config_mail = $this->_ctrl->config->get('config_mail') ) && is_array( $config_mail ) && isset( $config_mail['protocol'] ) ) {
				$mail = new Mail($this->_ctrl->config->get('config_mail'));
			} else {
				$mail = new Mail();
				$mail->protocol = $this->_ctrl->config->get('config_mail_protocol');
				$mail->parameter = $this->_ctrl->config->get('config_mail_parameter');
				$mail->smtp_hostname = $this->_ctrl->config->get('config_mail_smtp_hostname')?$this->_ctrl->config->get('config_mail_smtp_hostname'):$this->_ctrl->config->get('config_mail_smtp_host');
				$mail->smtp_username = $this->_ctrl->config->get('config_mail_smtp_username');
				$mail->smtp_password = html_entity_decode($this->_ctrl->config->get('config_mail_smtp_password'), ENT_QUOTES, 'UTF-8');
				$mail->smtp_port = $this->_ctrl->config->get('config_mail_smtp_port');
				$mail->smtp_timeout = $this->_ctrl->config->get('config_mail_smtp_timeout');
			}

			$mail->setTo($order_info['email']);
			$mail->setFrom($this->_ctrl->config->get('config_email'));
			$mail->setSender($store_name);
			$mail->setSubject( html_entity_decode($subject, ENT_QUOTES, 'UTF-8') );
			$mail->setHtml( html_entity_decode($message, ENT_QUOTES, 'UTF-8') );
			
			file_put_contents(DIR_SYSTEM . 'library/mreward_points/logs.log', print_r($mail, true),FILE_APPEND);
			
			$mail->send();
		}
	}
	
	public function addRewardsPointsByGenerator( $order_id, $order_status_id ) {
		$this->_ctrl->language->load('account/order');
		$this->_ctrl->load->model( 'account/customer' );
		
		/* @var $config array */
		if( null == ( $config = $this->_ctrl->config->get( 'mreward_points_settings' ) ) ) {
			return;
		}
		
		/* @var $order array */
		if( null == ( $order = $this->_ctrl->model_checkout_order->getOrder( $order_id ) ) ) {
			return;
		}

		/* @var $mrp_bonuses array */
		if( null != ( $mrp_bonuses = $this->_ctrl->config->get( 'mreward_points_bonuses' ) ) ) {
			/* @var $reason_code string */
			$reason_code = 'on_first_order';
			
			// add points if first order
			if( 
				! empty( $mrp_bonuses['on_first_order'] ) 
					&& 
				$order['customer_id'] 
					&& 
				in_array( $order_status_id, $config['order_status_add'] ) 
					&& 
				! $this->getTotalCustomerRewardsByReasonCode( $reason_code, $order['customer_id'] ) 
					&&
				! $this->getTotalCustomerCompleteOrders( $order['customer_id'] )
			) {
				/* @var $description string */
				$description = ! empty( $mrp_bonuses['text_on_first_order'][$order['language_id']]['content'] ) ? $mrp_bonuses['text_on_first_order'][$order['language_id']]['content'] : 'BONUS';

				$this->_ctrl->db->query("INSERT INTO `" . DB_PREFIX . "customer_reward` SET 
					`customer_id` = '" . (int)$order['customer_id'] . "', 
					`order_id`    = '0',
					`description` = '" . $this->_ctrl->db->escape( $description ). "', 
					`points`      = '" . (int) $mrp_bonuses['on_first_order'] . "', 
					`reason_code` = '" . $this->_ctrl->db->escape( $reason_code ) . "', 
					`date_added` = NOW()"
			   );
			}

			if( ! empty($config['order_status_remove']) && in_array($order_status_id, $config['order_status_remove']) ) {
				/* @var $order_status array */
				$order_status = $this->getOrderStatus($order_status_id);

				/* @var $new_description string */
				$new_description = $this->_ctrl->language->get('text_order_id') . ' #' . $order_id .' - '. $order_status['name'];

				$this->_ctrl->db->query("UPDATE `" . DB_PREFIX . "customer_reward` SET 
						`description` = '" . $this->_ctrl->db->escape( $new_description ). "', 
						`points`      = '0' 
					WHERE 
						`customer_id` = '" . $order['customer_id'] . "' 
					AND 
						order_id = '" . $order_id . "'"
			   );
			}
		}

		if( empty( $config['order_status_add'] ) || !in_array($order_status_id, $config['order_status_add']) ){
			return;
		}

		if( ! $order['customer_id'] || $this->getTotalCustomerRewardsByOrderId( $order_id ) ) {
			return;
		}

		$this->addReward(
			$order['customer_id'], 
			$this->_ctrl->language->get('text_order_id') . ' #' . $order_id, 
			$this->getTotalRewardsFromOrderProductsByOrderId( $order_id ), 
			$order_id
		);
	}
	
	public function addPrefixSuffix($points) {
		$mrp_settings = $this->_ctrl->config->get( 'mreward_points_settings' );
		$points_string  = '';

		$points_string .= !empty( $mrp_settings['bold_points_number']) ? '<b>' : '';

		if (!empty( $mrp_settings['points_prefix'])) {
			$points_string .= $mrp_settings['points_prefix'];
		}
		$points_string .= (int)$points;
		if ( !empty( $mrp_settings['points_suffix'] ) ) {
			$points_string .= $mrp_settings['points_suffix'];
		}
		$points_string .= !empty( $mrp_settings['bold_points_number']) ? '</b>' : '';
		return $points_string;
	}
	
	public function reviewSave( $review_id, $data ) {
		$mrp_bonuses = $this->_ctrl->config->get( 'mreward_points_bonuses' );
		//Auto approve reviews 
		if( isset( $data['rating'] ) && $mrp_bonuses['auto_approve_reviews'] && $data['rating'] >= $mrp_bonuses['auto_approve_reviews']) {
			$this->_ctrl->db->query("UPDATE " . DB_PREFIX . "review SET `status` = '1' WHERE review_id = '" . (int) $review_id . "'"); 
		}
		//Add bonus points on add product review
		if( $mrp_bonuses['on_product_review'] && $this->_ctrl->customer->isLogged() ) {
			$check = $this->_ctrl->db->query("SELECT COUNT(*) AS total_user_reviews FROM " . DB_PREFIX . "review WHERE `customer_id` = '" . (int)$this->_ctrl->customer->getId() . "'");
			
			if( $mrp_bonuses['product_review_limit'] == 0 || $check->row['total_user_reviews'] <= $mrp_bonuses['product_review_limit'] ) {

				$new_description = !empty( $mrp_bonuses['text_on_product_review'][(int)$this->_ctrl->config->get('config_language_id')]['content'] ) ? $mrp_bonuses['text_on_product_review'][(int)$this->_ctrl->config->get('config_language_id')]['content'] : 'BONUS';

				$this->_ctrl->db->query("INSERT INTO " . DB_PREFIX . "customer_reward SET 
					 `customer_id` = '" . (int)$this->_ctrl->customer->getId() . "', 
					 `order_id`    = '0',
					 `description` = '" . $this->_ctrl->db->escape( $new_description ). "', 
					 `points`      = '" . $mrp_bonuses['on_product_review']. "', 
					 `reason_code` = 'on_product_review', 
					 `date_added` = NOW()"
				);
			}
		}
	}
	
	public function customerNewsletter( $customer_id, $data ) {
		$mrp_bonuses = $this->_ctrl->config->get( 'mreward_points_bonuses' );

		if( $mrp_bonuses['on_registration'] ) {
			$new_description = !empty( $mrp_bonuses['text_on_registration'][(int)$this->_ctrl->config->get('config_language_id')]['content'] ) ? $mrp_bonuses['text_on_registration'][(int)$this->_ctrl->config->get('config_language_id')]['content'] : 'BONUS';

			$this->_ctrl->db->query("INSERT INTO " . DB_PREFIX . "customer_reward SET 
				 `customer_id` = '" . $customer_id . "', 
				 `order_id`    = '0', 
				 `description` = '" . $this->_ctrl->db->escape( $new_description ). "', 
				 `points`      = '" . $mrp_bonuses['on_registration']. "', 
				 `reason_code` = 'on_registration', 
				 `date_added` = NOW()"
			);
		}
		// Newsletter bonus
		if( $mrp_bonuses['on_newsletter'] && ( isset( $data['newsletter'] ) && (int)$data['newsletter'] ) ) {
			$new_description = !empty( $mrp_bonuses['text_on_newsletter'][(int)$this->_ctrl->config->get('config_language_id')]['content'] ) ? $mrp_bonuses['text_on_newsletter'][(int)$this->_ctrl->config->get('config_language_id')]['content'] : 'BONUS';

			$this->_ctrl->db->query("INSERT INTO " . DB_PREFIX . "customer_reward SET 
				 `customer_id` = '" . $customer_id . "', 
				 `order_id`    = '0',
				 `description` = '" . $this->_ctrl->db->escape( $new_description ). "', 
				 `points`      = '" . $mrp_bonuses['on_newsletter']. "', 
				 `reason_code` = 'on_newsletter', 
				 `date_added` = NOW()"
			);
		}
	}
	
	public function customerNewsletterUnsubscribe( $newsletter ) {
		$mrp_bonuses = $this->_ctrl->config->get( 'mreward_points_bonuses' );

		if ($mrp_bonuses['on_newsletter'] && (int)$newsletter) {
			$check = $this->_ctrl->db->query("SELECT * FROM `" . DB_PREFIX . "customer_reward` WHERE `customer_id` = '" . (int)$this->_ctrl->customer->getId() . "' AND `reason_code` = 'on_newsletter'");
			
			if (!$check->num_rows) {
				$new_description = !empty( $mrp_bonuses['text_on_newsletter'][(int)$this->_ctrl->config->get('config_language_id')]['content'] ) ? $mrp_bonuses['text_on_newsletter'][(int)$this->_ctrl->config->get('config_language_id')]['content'] : 'BONUS';

				$this->_ctrl->db->query("INSERT INTO `" . DB_PREFIX . "customer_reward` SET 
					 `customer_id` = '" . (int)$this->_ctrl->customer->getId() . "', 
					 `order_id`    = '0',
					 `description` = '" . $this->_ctrl->db->escape( $new_description ). "', 
					 `points`      = '" . $mrp_bonuses['on_newsletter']. "', 
					 `reason_code` = 'on_newsletter', 
					 `date_added` = NOW()"
				);
			}
		} elseif ( $mrp_bonuses['on_newsletter'] && $mrp_bonuses['newsletter_unsub'] && (int)$newsletter == 0) {
			$this->_ctrl->db->query("DELETE FROM `" . DB_PREFIX . "customer_reward` WHERE `customer_id` = '" . (int)$this->_ctrl->customer->getId() . "' AND `reason_code` = 'on_newsletter'");
		}
	}
}